#!/usr/bin/env python3
import json
import os
import copy
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext
import webbrowser

def load_language_file(path):
    """Lädt eine einfache key=value Sprachdatei."""
    data = {}
    if not os.path.exists(path):
        return data
    with open(path, "r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith("#"):
                continue
            if "=" not in line:
                continue
            key, value = line.split("=", 1)
            data[key.strip()] = value.strip()
    return data

class MaestroCueGUI(tk.Tk):
    def __init__(self):
        super().__init__()

        # Basisverzeichnis
        base_dir = os.path.dirname(os.path.abspath(__file__))

        # Pfad zur Config-Datei
        self.config_path = os.path.join(base_dir, "config.cfg")

        # Sprachen vorbereiten
        self.lang_files = {
            "de": os.path.join(base_dir, "de.cfg"),
            "en": os.path.join(base_dir, "en.cfg"),
            "fr": os.path.join(base_dir, "fr.cfg"),
            "es": os.path.join(base_dir, "es.cfg"),
        }
        self.languages = {
            code: load_language_file(path)
            for code, path in self.lang_files.items()
        }

        # Zuletzt gewählte Sprache aus config.cfg holen (falls vorhanden)
        last_lang = self._load_last_language()

        if last_lang in self.languages:
            self.current_lang = last_lang
        elif "de" in self.languages:
            self.current_lang = "de"
        else:
            self.current_lang = next(iter(self.languages))

        self.tr = self.languages.get(self.current_lang, {})

        # Pfad- und Statusvariablen
        self.source_path = tk.StringVar()
        self.target_path = tk.StringVar()

        self.source_data = None
        self.source_cues = []
        self.target_data = None

        self.source_valid = False
        self.target_valid = False

        # Show-Namen und IDs der Quelle
        self.source_show_name = ""
        self.source_show_id = ""

        # Ziel-Show-Metadaten
        self.target_show_name = tk.StringVar()
        self.target_show_id = tk.StringVar()
        self.target_show_desc = tk.StringVar()

        # Für Checkbuttons
        self.cue_vars = []

        # GUI bauen
        self._build_widgets()
        self.apply_translations()

    def _load_last_language(self):
        """Liest die zuletzt gewählte Sprache aus config.cfg (language=xx)."""
        try:
            if not os.path.exists(self.config_path):
                return None
            with open(self.config_path, "r", encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith("#"):
                        continue
                    if "=" not in line:
                        continue
                    key, value = line.split("=", 1)
                    if key.strip().lower() == "language":
                        return value.strip()
        except Exception:
            # Fehler beim Lesen ignorieren -> Fallback greift
            return None
        return None

    def _save_last_language(self):
        """Speichert die aktuell gewählte Sprache in config.cfg."""
        try:
            with open(self.config_path, "w", encoding="utf-8") as f:
                f.write(f"language={self.current_lang}\n")
        except Exception:
            # Absichtlich stumm – das Tool soll weiterlaufen,
            # auch wenn die Config nicht geschrieben werden kann.
            pass

    # -------------------------------------------------------------------------
    # Übersetzungshilfen
    # -------------------------------------------------------------------------
    def t(self, msg_key, **kwargs):
        """Übersetzung holen, bei Bedarf mit .format(**kwargs)."""
        text = self.tr.get(msg_key, msg_key)
        # \n aus Sprachdatei in echte Zeilenumbrüche umwandeln
        text = text.replace("\\n", "\n")
        if kwargs:
            try:
                text = text.format(**kwargs)
            except Exception:
                pass
        return text


    def set_language(self, lang_code):
        if lang_code not in self.languages:
            return
        self.current_lang = lang_code
        self.tr = self.languages.get(lang_code, {})
        # Dropdown-Wert aktualisieren (falls set_language später mal
        # auch intern/programmgesteuert aufgerufen wird)
        if hasattr(self, "lang_var"):
            self.lang_var.set(lang_code)
        self.apply_translations()
        # Sprache in config.cfg persistieren
        self._save_last_language()

    def apply_translations(self):
        # Fenster-Titel
        self.title(self.t("app_title"))

        # Labels oben
        self.lbl_source_file.config(text=self.t("label_source_show_json"))
        self.lbl_target_file.config(text=self.t("label_target_show_json"))

        # Hinweis unter den Dateifeldern
        self.hint_label.config(text=self.t("hint_under_paths"))

        # Frames
        self.frame_cues.config(text=self.t("frame_cues_title"))
        self.frame_target_meta.config(text=self.t("frame_target_meta_title"))
        self.frame_log.config(text=self.t("frame_log_title"))

        # Quellshow-Labels (wenn noch nichts geladen: Default-Texte)
        if not self.source_show_name:
            self.lbl_source_show_name.config(text=self.t("source_show_name_not_loaded"))
        else:
            self.lbl_source_show_name.config(
                text=f"{self.t('source_show_name_label')} {self.source_show_name}"
            )

        if not self.source_show_id:
            self.lbl_source_show_id.config(text=self.t("source_show_id_not_loaded"))
        else:
            self.lbl_source_show_id.config(
                text=f"{self.t('source_show_id_label')} {self.source_show_id}"
            )

        # Ziel-Metadaten-Labels
        self.lbl_target_show_name.config(text=self.t("target_show_name_label"))
        self.lbl_target_show_id.config(text=self.t("target_show_id_label"))
        self.lbl_target_show_desc.config(text=self.t("target_show_desc_label"))

        # Buttons
        self.btn_save.config(text=self.t("button_save"))
        self.btn_quit.config(text=self.t("button_quit"))

        # Sprache-Label
        self.lbl_lang.config(text="Sprache:" if self.current_lang == "de" else "Language:")

        # Hilfelink-Text aus Sprachdatei
        self.link_label.config(text=self.t("help_link_label"))

    # -------------------------------------------------------------------------
    # Klick-Handler für Hilfelink
    # -------------------------------------------------------------------------
    def on_help_link_clicked(self, event=None):
        url = self.t("help_link_url")  # kommt aus lang_*.cfg
        if url and url.startswith("http"):
            webbrowser.open(url)

    # -------------------------------------------------------------------------
    # GUI-Struktur
    # -------------------------------------------------------------------------
    def _build_widgets(self):
        # --- Pfad-Auswahl oben ------------------------------------------------
        frame_paths = tk.Frame(self)
        frame_paths.pack(fill="x", padx=10, pady=10)

        # Sprachwahl (rechts oben)
        self.lang_var = tk.StringVar(value=self.current_lang)
        self.lbl_lang = tk.Label(frame_paths, text="Sprache:")
        self.lbl_lang.grid(row=0, column=3, sticky="e", padx=(20, 5))

        lang_options = list(self.languages.keys())
        self.opt_lang = tk.OptionMenu(frame_paths, self.lang_var, *lang_options, command=self.set_language)
        self.opt_lang.grid(row=0, column=4, sticky="w")

        # Quell-Show
        self.lbl_source_file = tk.Label(frame_paths, text="")
        self.lbl_source_file.grid(row=0, column=0, sticky="w")
        tk.Entry(frame_paths, textvariable=self.source_path, width=60).grid(row=0, column=1, padx=5)
        self.btn_browse_source = tk.Button(frame_paths, text="...", command=self.browse_source)
        self.btn_browse_source.grid(row=0, column=2, padx=5)

        # Ziel-Show
        self.lbl_target_file = tk.Label(frame_paths, text="")
        self.lbl_target_file.grid(row=1, column=0, sticky="w", pady=(5, 0))
        tk.Entry(frame_paths, textvariable=self.target_path, width=60).grid(row=1, column=1, padx=5, pady=(5, 0))
        self.btn_browse_target = tk.Button(frame_paths, text="...", command=self.browse_target)
        self.btn_browse_target.grid(row=1, column=2, padx=5, pady=(5, 0))

        # Hinweis unter den Dateifeldern (aus Sprachdatei, linksbündig)
        self.hint_label = tk.Label(
            self,
            text="",  # wird in apply_translations gesetzt
            fg="gray",
            anchor="w",
            justify="left"
        )
        self.hint_label.pack(fill="x", padx=10, pady=(0, 10))

        # --- Info-Bereich für Quell-Show (Name + ID) über der Cue-Liste ------
        self.frame_source_info = tk.Frame(self)
        self.frame_source_info.pack(fill="x", padx=10, pady=(0, 5))

        self.lbl_source_show_name = tk.Label(
            self.frame_source_info,
            text="",
            anchor="w",
            justify="left"
        )
        self.lbl_source_show_name.pack(fill="x", padx=5, pady=(0, 0))

        self.lbl_source_show_id = tk.Label(
            self.frame_source_info,
            text="",
            anchor="w",
            justify="left"
        )
        self.lbl_source_show_id.pack(fill="x", padx=5, pady=(0, 0))

        # --- Bereich für Cues aus der Quellshow -------------------------------
        self.frame_cues = tk.LabelFrame(self, text="")
        self.frame_cues.pack(fill="both", expand=True, padx=10, pady=5)

        # Scrollbarer Bereich
        self.cue_canvas = tk.Canvas(self.frame_cues)
        self.cue_scrollbar = tk.Scrollbar(self.frame_cues, orient="vertical", command=self.cue_canvas.yview)
        self.cue_inner_frame = tk.Frame(self.cue_canvas)

        self.cue_inner_frame.bind(
            "<Configure>",
            lambda e: self.cue_canvas.configure(
                scrollregion=self.cue_canvas.bbox("all")
            )
        )

        self.cue_canvas.create_window((0, 0), window=self.cue_inner_frame, anchor="nw")
        self.cue_canvas.configure(yscrollcommand=self.cue_scrollbar.set)

        self.cue_canvas.pack(side="left", fill="both", expand=True)
        self.cue_scrollbar.pack(side="right", fill="y")

        # --- Bereich für Ziel-Show-Name, ID, Beschreibung ---------------------
        self.frame_target_meta = tk.LabelFrame(self, text="")
        self.frame_target_meta.pack(fill="x", padx=10, pady=5)

        # Name
        self.lbl_target_show_name = tk.Label(self.frame_target_meta, text="")
        self.lbl_target_show_name.grid(row=0, column=0, sticky="w", padx=5, pady=(5, 2))
        tk.Entry(self.frame_target_meta, textvariable=self.target_show_name, width=50).grid(
            row=0, column=1, sticky="we", padx=5, pady=(5, 2)
        )

        # ID
        self.lbl_target_show_id = tk.Label(self.frame_target_meta, text="")
        self.lbl_target_show_id.grid(row=1, column=0, sticky="w", padx=5, pady=(2, 2))
        tk.Entry(self.frame_target_meta, textvariable=self.target_show_id, width=50).grid(
            row=1, column=1, sticky="we", padx=5, pady=(2, 2)
        )

        # Beschreibung (description)
        self.lbl_target_show_desc = tk.Label(self.frame_target_meta, text="")
        self.lbl_target_show_desc.grid(row=2, column=0, sticky="w", padx=5, pady=(2, 5))
        tk.Entry(self.frame_target_meta, textvariable=self.target_show_desc, width=50).grid(
            row=2, column=1, sticky="we", padx=5, pady=(2, 5)
        )

        self.frame_target_meta.columnconfigure(1, weight=1)

        # --- Unterer Bereich: Log + Buttons ----------------------------------
        frame_bottom = tk.Frame(self)
        frame_bottom.pack(fill="both", expand=False, padx=10, pady=5)

        # Log-Ausgabe
        self.frame_log = tk.LabelFrame(frame_bottom, text="")
        self.frame_log.pack(side="left", fill="both", expand=True, padx=(0, 5))

        self.log_text = scrolledtext.ScrolledText(self.frame_log, wrap="word", height=8)
        self.log_text.pack(fill="both", expand=True)

        # Buttons
        frame_buttons = tk.Frame(frame_bottom)
        frame_buttons.pack(side="right", fill="y")

        self.btn_save = tk.Button(
            frame_buttons,
            text="",
            command=self.save_selected_cues,
            state="disabled"
        )
        self.btn_save.pack(fill="x", pady=(0, 5))

        self.btn_quit = tk.Button(frame_buttons, text="", command=self.destroy)
        self.btn_quit.pack(fill="x")

        # --- Footer ganz unten: Author links, Link rechts --------------------
        footer = tk.Frame(self)
        footer.pack(fill="x", padx=10, pady=(0, 10))

        # Author-Text (links)
        author_label = tk.Label(
            footer,
            text="(c) Christian Haubitz-Reinke & ChatGPT 5.1 - 23.12.2025 (v0.91)",
            fg="gray"
        )
        author_label.pack(side="left", anchor="w")

        # Hilfelink (rechts)
        self.link_label = tk.Label(
            footer,
            text="",          # Text kommt aus der Sprachdatei
            fg="blue",
            cursor="hand2"
        )
        self.link_label.pack(side="right", anchor="e")

        # Klick-Handler
        self.link_label.bind("<Button-1>", self.on_help_link_clicked)

    # -------------------------------------------------------------------------
    # Datei-Dialoge
    # -------------------------------------------------------------------------
    def browse_source(self):
        path = filedialog.askopenfilename(
            title=self.t("label_source_show_json"),
            filetypes=[("JSON-Dateien", "*.json"), ("Alle Dateien", "*.*")]
        )
        if path:
            self.source_path.set(path)
            self.load_source_show(path)

    def browse_target(self):
        path = filedialog.askopenfilename(
            title=self.t("label_target_show_json"),
            filetypes=[("JSON-Dateien", "*.json"), ("Alle Dateien", "*.*")]
        )
        if path:
            self.target_path.set(path)
            self.load_target_show(path)

    # -------------------------------------------------------------------------
    # JSON laden und validieren
    # -------------------------------------------------------------------------
    def load_source_show(self, path: str):
        self.log_text.insert(tk.END, self.t("log_loading_source", path=path) + "\n")
        self.log_text.see(tk.END)

        try:
            with open(path, "r", encoding="utf-8") as f:
                data = json.load(f)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                self.t("msg_error_source_load", error=e)
            )
            self.source_data = None
            self.source_cues = []
            self.source_valid = False
            self.source_show_name = ""
            self.source_show_id = ""
            self.lbl_source_show_name.config(text=self.t("source_show_name_error"))
            self.lbl_source_show_id.config(text=self.t("source_show_id_error"))
            self.clear_cue_list()
            self.update_save_button_state()
            return

        show = data.get("show")
        if not isinstance(show, dict):
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                self.t("msg_error_source_structure")
            )
            self.source_data = None
            self.source_cues = []
            self.source_valid = False
            self.source_show_name = ""
            self.source_show_id = ""
            self.lbl_source_show_name.config(text=self.t("source_show_name_invalid"))
            self.lbl_source_show_id.config(text=self.t("source_show_id_invalid"))
            self.clear_cue_list()
            self.update_save_button_state()
            return

        self.source_show_name = show.get("name", "<ohne Namen>")
        self.source_show_id = show.get("ID") or show.get("id") or "<ohne ID>"

        self.lbl_source_show_name.config(
            text=f"{self.t('source_show_name_label')} {self.source_show_name}"
        )
        self.lbl_source_show_id.config(
            text=f"{self.t('source_show_id_label')} {self.source_show_id}"
        )

        pattern_cue = show.get("patternCue")
        if not isinstance(pattern_cue, list):
            pattern_cue = []
            show["patternCue"] = pattern_cue

        self.source_data = data
        self.source_cues = pattern_cue
        self.source_valid = True

        self.log_text.insert(
            tk.END,
            self.t(
                "log_source_loaded",
                name=self.source_show_name,
                id=self.source_show_id,
                count=len(self.source_cues)
            ) + "\n\n"
        )
        self.log_text.see(tk.END)

        self.populate_cue_list()
        self.update_save_button_state()

    def load_target_show(self, path: str):
        self.log_text.insert(tk.END, self.t("log_loading_target", path=path) + "\n")
        self.log_text.see(tk.END)

        try:
            with open(path, "r", encoding="utf-8") as f:
                data = json.load(f)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                self.t("msg_error_target_load", error=e)
            )
            self.target_data = None
            self.target_valid = False
            self.target_show_name.set("")
            self.target_show_id.set("")
            self.target_show_desc.set("")
            self.update_save_button_state()
            return

        show = data.get("show")
        if not isinstance(show, dict):
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                self.t("msg_error_target_structure")
            )
            self.target_data = None
            self.target_valid = False
            self.target_show_name.set("")
            self.target_show_id.set("")
            self.target_show_desc.set("")
            self.update_save_button_state()
            return

        pattern_cue = show.get("patternCue")
        if not isinstance(pattern_cue, list):
            show["patternCue"] = []

        # Name
        name = show.get("name", "")
        if isinstance(name, str):
            self.target_show_name.set(name)
        else:
            self.target_show_name.set("")

        # ID
        target_id_val = show.get("ID") or show.get("id") or ""
        if isinstance(target_id_val, str):
            self.target_show_id.set(target_id_val)
        else:
            self.target_show_id.set("")

        # Beschreibung
        desc_val = show.get("description", "")
        if isinstance(desc_val, str):
            self.target_show_desc.set(desc_val)
        else:
            self.target_show_desc.set("")

        self.target_data = data
        self.target_valid = True

        self.log_text.insert(
            tk.END,
            self.t(
                "log_target_valid",
                name=self.target_show_name.get() or "(leer)",
                id=self.target_show_id.get() or "(leer)"
            ) + "\n\n"
        )
        self.log_text.see(tk.END)

        self.update_save_button_state()

    # -------------------------------------------------------------------------
    # Cue-Liste
    # -------------------------------------------------------------------------
    def clear_cue_list(self):
        for child in self.cue_inner_frame.winfo_children():
            child.destroy()
        self.cue_vars.clear()

    def populate_cue_list(self):
        self.clear_cue_list()

        if not self.source_cues:
            lbl = tk.Label(self.cue_inner_frame, text=self.t("no_cues_in_source"))
            lbl.pack(anchor="w", padx=5, pady=5)
            return

        for idx, cue in enumerate(self.source_cues):
            name = cue.get("name", "<ohne Name>")
            uuid = cue.get("uuid", "<ohne UUID>")
            duration = cue.get("duration", "")

            text = f"[{idx}] {name} | UUID: {uuid}"
            if duration != "":
                text += f" | Dauer: {duration}"

            var = tk.BooleanVar(value=False)
            chk = tk.Checkbutton(self.cue_inner_frame, text=text, variable=var, anchor="w", justify="left")
            chk.pack(fill="x", anchor="w", padx=5, pady=2)

            self.cue_vars.append(var)

    # -------------------------------------------------------------------------
    # Save-Button Zustand
    # -------------------------------------------------------------------------
    def update_save_button_state(self):
        if self.source_valid and self.target_valid:
            self.btn_save.config(state="normal")
        else:
            self.btn_save.config(state="disabled")

    # -------------------------------------------------------------------------
    # Speichern
    # -------------------------------------------------------------------------
    def save_selected_cues(self):
        if not (self.source_valid and self.target_valid):
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                self.t("msg_error_no_source_target")
            )
            return

        selected_indices = [i for i, var in enumerate(self.cue_vars) if var.get()]
        if not selected_indices:
            messagebox.showwarning(
                self.t("msg_warn_no_cues_selected_title"),
                self.t("msg_warn_no_cues_selected")
            )
            return

        self.log_text.insert(
            tk.END,
            self.t("log_selected_cues", indices=selected_indices) + "\n"
        )
        self.log_text.see(tk.END)

        show = self.target_data.get("show")
        if show is None:
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                self.t("msg_error_missing_show")
            )
            return

        pattern_cue_target = show.get("patternCue")
        if not isinstance(pattern_cue_target, list):
            pattern_cue_target = []
            show["patternCue"] = pattern_cue_target

        # Name
        new_name = self.target_show_name.get().strip()
        if new_name:
            show["name"] = new_name
            self.log_text.insert(
                tk.END,
                self.t("log_target_name_set", name=new_name) + "\n"
            )
        else:
            self.log_text.insert(
                tk.END,
                self.t("log_target_name_not_changed") + "\n"
            )

        # ID
        new_id = self.target_show_id.get().strip()
        if new_id:
            if "ID" in show:
                id_key = "ID"
            elif "id" in show:
                id_key = "id"
            else:
                id_key = "ID"

            show[id_key] = new_id
            self.log_text.insert(
                tk.END,
                self.t("log_target_id_set", id=new_id, key=id_key) + "\n"
            )
        else:
            self.log_text.insert(
                tk.END,
                self.t("log_target_id_not_changed") + "\n"
            )

        # Beschreibung
        new_desc = self.target_show_desc.get().strip()
        if new_desc:
            show["description"] = new_desc
            self.log_text.insert(
                tk.END,
                self.t("log_target_desc_set") + "\n"
            )
        else:
            self.log_text.insert(
                tk.END,
                self.t("log_target_desc_not_changed") + "\n"
            )

        # Cues kopieren
        copied_count = 0
        for idx in selected_indices:
            if 0 <= idx < len(self.source_cues):
                cue_copy = copy.deepcopy(self.source_cues[idx])
                pattern_cue_target.append(cue_copy)
                copied_count += 1

        target_path = self.target_path.get().strip()
        try:
            with open(target_path, "w", encoding="utf-8") as f:
                json.dump(self.target_data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_source_load_title"),
                str(e)
            )
            return

        msg = self.t("msg_info_success", count=copied_count)
        self.log_text.insert(
            tk.END,
            self.t("log_finished", count=copied_count) + "\n\n"
        )
        self.log_text.see(tk.END)
        messagebox.showinfo(
            self.t("msg_info_success_title"),
            msg
        )

def main():
    app = MaestroCueGUI()
    app.mainloop()


if __name__ == "__main__":
    main()
