#!/usr/bin/env python3
import json
import os
import copy
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext
import webbrowser

# Gruppen 1–4 ↔ Parameternamen im JSON
GROUP_KEYS = {
    1: "params",
    2: "secondaryParams",
    3: "tertiaryParams",
    4: "quaternaryParams",
}


def load_language_file(path):
    """Lädt eine einfache key=value Sprachdatei."""
    data = {}
    if not os.path.exists(path):
        return data
    with open(path, "r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith("#"):
                continue
            if "=" not in line:
                continue
            key, value = line.split("=", 1)
            data[key.strip()] = value.strip()
    return data


class MaestroGroupRemapGUI(tk.Tk):
    def __init__(self):
        super().__init__()

        # Basisverzeichnis
        base_dir = os.path.dirname(os.path.abspath(__file__))

        # Pfad zur Config-Datei (Sprache)
        self.config_path = os.path.join(base_dir, "config.cfg")

        # Sprachdateien
        self.lang_files = {
            "de": os.path.join(base_dir, "de.cfg"),
            "en": os.path.join(base_dir, "en.cfg"),
            "fr": os.path.join(base_dir, "fr.cfg"),
            "es": os.path.join(base_dir, "es.cfg"),
        }
        self.languages = {
            code: load_language_file(path)
            for code, path in self.lang_files.items()
        }

        # zuletzt verwendete Sprache laden
        last_lang = self._load_last_language()
        if last_lang in self.languages:
            self.current_lang = last_lang
        elif "de" in self.languages:
            self.current_lang = "de"
        else:
            self.current_lang = next(iter(self.languages)) if self.languages else "de"

        self.tr = self.languages.get(self.current_lang, {})

        self.source_path = tk.StringVar()
        self.show_data = None
        self.show_obj = None  # self.show_data["show"]

        # Show-Metadaten
        self.show_name = tk.StringVar()
        self.show_id = tk.StringVar()
        self.show_desc = tk.StringVar()

        # Mapping-Variablen
        self.group_vars = {}

        # GUI
        self._build_widgets()
        self.apply_translations()

    # -------------------------------------------------------------------------
    # Config Sprache
    # -------------------------------------------------------------------------
    def _load_last_language(self):
        """Liest die zuletzt gewählte Sprache aus config.cfg (language=xx)."""
        try:
            if not os.path.exists(self.config_path):
                return None
            with open(self.config_path, "r", encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith("#"):
                        continue
                    if "=" not in line:
                        continue
                    key, value = line.split("=", 1)
                    if key.strip().lower() == "language":
                        return value.strip()
        except Exception:
            return None
        return None

    def _save_last_language(self):
        """Speichert die aktuell gewählte Sprache in config.cfg."""
        try:
            with open(self.config_path, "w", encoding="utf-8") as f:
                f.write(f"language={self.current_lang}\n")
        except Exception:
            # Tool soll auch laufen, wenn Config nicht geschrieben werden kann.
            pass

    # -------------------------------------------------------------------------
    # Übersetzungshilfen
    # -------------------------------------------------------------------------
    def t(self, msg_key, **kwargs):
        """Übersetzung holen, bei Bedarf mit .format(**kwargs)."""
        text = self.tr.get(msg_key, msg_key)
        # \n aus Sprachdatei in echte Zeilenumbrüche umwandeln
        text = text.replace("\\n", "\n")
        if kwargs:
            try:
                text = text.format(**kwargs)
            except Exception:
                pass
        return text


    def set_language(self, lang_code):
        if lang_code not in self.languages:
            return
        self.current_lang = lang_code
        self.tr = self.languages.get(lang_code, {})
        if hasattr(self, "lang_var"):
            self.lang_var.set(lang_code)
        self.apply_translations()
        self._save_last_language()

    def apply_translations(self):
        # Fenster-Titel
        self.title(self.t("app_title_groups"))

        # Pfad-Label & Hinweis
        self.lbl_source_file.config(text=self.t("label_source_show_json_groups"))

        self.hint_label.config(text=self.t("hint_groups_info"))

        # Show-Meta-Frame
        self.frame_meta.config(text=self.t("frame_meta_title_groups"))
        self.lbl_show_name.config(text=self.t("show_name_label"))
        self.lbl_show_id.config(text=self.t("show_id_label"))
        self.lbl_show_desc.config(text=self.t("show_desc_label"))

        # Mapping-Frame
        self.frame_map.config(text=self.t("frame_map_title_groups"))
        self.lbl_map_info.config(text=self.t("map_info_text"))
        self.lbl_map_hint.config(text=self.t("map_hint_text"))

        for i in range(1, 5):
            self.lbl_map_group[i].config(
                text=self.t("map_group_line", new=i)
            )

        # Log-Frame
        self.frame_log.config(text=self.t("frame_log_title"))

        # Buttons
        self.btn_save.config(text=self.t("button_save_as"))
        self.btn_quit.config(text=self.t("button_quit"))

        # Sprachlabel
        if self.current_lang == "de":
            self.lbl_lang.config(text="Sprache:")
        else:
            self.lbl_lang.config(text="Language:")

        # Footer
        self.author_label.config(text=self.t("footer_author_groups"))
        self.link_label.config(text=self.t("help_link_label"))

    # -------------------------------------------------------------------------
    # Klick-Handler für Hilfelink
    # -------------------------------------------------------------------------
    def on_help_link_clicked(self, event=None):
        url = self.t("help_link_url")  # kommt aus lang_*.cfg
        if url and url.startswith("http"):
            webbrowser.open(url)

    # ------------------------------------------------------------------
    # GUI-Aufbau
    # ------------------------------------------------------------------
    def _build_widgets(self):
        # Obere Zeile: Datei + Sprache
        frame_paths = tk.Frame(self)
        frame_paths.pack(fill="x", padx=10, pady=10)

        # Label + Pfad Show
        self.lbl_source_file = tk.Label(frame_paths, text="")
        self.lbl_source_file.grid(row=0, column=0, sticky="w")
        tk.Entry(frame_paths, textvariable=self.source_path, width=60).grid(
            row=0, column=1, padx=5
        )
        tk.Button(
            frame_paths,
            text="…",
            width=3,
            command=self.browse_source
        ).grid(row=0, column=2, padx=5)

        # Sprachwahl (rechts oben)
        self.lang_var = tk.StringVar(value=self.current_lang)
        self.lbl_lang = tk.Label(frame_paths, text="Sprache:")
        self.lbl_lang.grid(row=0, column=3, sticky="e", padx=(20, 5))

        lang_options = list(self.languages.keys())
        self.opt_lang = tk.OptionMenu(frame_paths, self.lang_var, *lang_options, command=self.set_language)
        self.opt_lang.grid(row=0, column=4, sticky="w")

        # Infozeile
        self.hint_label = tk.Label(
            self,
            text="",
            fg="gray",
            anchor="w",
            justify="left"
        )
        self.hint_label.pack(fill="x", padx=10, pady=(0, 10))

        # Show-Metadaten
        self.frame_meta = tk.LabelFrame(self, text="", padx=5, pady=5)
        self.frame_meta.pack(fill="x", padx=10, pady=5)

        self.lbl_show_name = tk.Label(self.frame_meta, text="")
        self.lbl_show_name.grid(row=0, column=0, sticky="w", padx=5, pady=2)
        tk.Entry(self.frame_meta, textvariable=self.show_name, width=50).grid(
            row=0, column=1, sticky="we", padx=5, pady=2
        )

        self.lbl_show_id = tk.Label(self.frame_meta, text="")
        self.lbl_show_id.grid(row=1, column=0, sticky="w", padx=5, pady=2)
        tk.Entry(self.frame_meta, textvariable=self.show_id, width=50).grid(
            row=1, column=1, sticky="we", padx=5, pady=2
        )

        self.lbl_show_desc = tk.Label(self.frame_meta, text="")
        self.lbl_show_desc.grid(row=2, column=0, sticky="w", padx=5, pady=2)
        tk.Entry(self.frame_meta, textvariable=self.show_desc, width=50).grid(
            row=2, column=1, sticky="we", padx=5, pady=2
        )

        self.frame_meta.columnconfigure(1, weight=1)

        # Mapping-Bereich
        self.frame_map = tk.LabelFrame(self, text="", padx=5, pady=5)
        self.frame_map.pack(fill="x", padx=10, pady=5)

        self.lbl_map_info = tk.Label(
            self.frame_map,
            text="",
            justify="left"
        )
        self.lbl_map_info.grid(row=0, column=0, columnspan=3, sticky="w", padx=5, pady=(0, 10))

        options = ["1", "2", "3", "4"]
        self.lbl_map_group = {}

        for i in range(1, 5):
            lbl = tk.Label(self.frame_map, text="")
            lbl.grid(row=i, column=0, sticky="w", padx=5, pady=2)
            self.lbl_map_group[i] = lbl

            var = tk.StringVar(value=str(i))  # Standard: 1→1, 2→2, …
            cb = tk.OptionMenu(self.frame_map, var, *options)
            cb.config(width=5)
            cb.grid(row=i, column=1, sticky="w", padx=5, pady=2)
            self.group_vars[i] = var

        self.lbl_map_hint = tk.Label(
            self.frame_map,
            text="",
            fg="gray"
        )
        self.lbl_map_hint.grid(row=5, column=0, columnspan=3, sticky="w", padx=5, pady=(10, 0))

        # Unterer Bereich: Log + Buttons
        frame_bottom = tk.Frame(self)
        frame_bottom.pack(fill="both", expand=True, padx=10, pady=5)

        # Log-Ausgabe
        self.frame_log = tk.LabelFrame(frame_bottom, text="", padx=5, pady=5)
        self.frame_log.pack(side="left", fill="both", expand=True)

        self.log_text = scrolledtext.ScrolledText(self.frame_log, wrap="word", height=10)
        self.log_text.pack(fill="both", expand=True)

        # Buttons rechts
        frame_buttons = tk.Frame(frame_bottom)
        frame_buttons.pack(side="right", fill="y", padx=(5, 0))

        self.btn_save = tk.Button(
            frame_buttons,
            text="",
            command=self.save_as,
            state="disabled"
        )
        self.btn_save.pack(fill="x", pady=(0, 5))

        self.btn_quit = tk.Button(
            frame_buttons,
            text="",
            command=self.destroy
        )
        self.btn_quit.pack(fill="x")

        # --- Footer ganz unten: Author links, Link rechts --------------------
        footer = tk.Frame(self)
        footer.pack(fill="x", padx=10, pady=(0, 10))

        # Author-Text (links)
        self.author_label = tk.Label(
            footer,
            text="",
            fg="gray"
        )
        self.author_label.pack(side="left", anchor="w")

        # Hilfelink (rechts)
        self.link_label = tk.Label(
            footer,
            text="",
            fg="blue",
            cursor="hand2"
        )
        self.link_label.pack(side="right", anchor="e")

        # Klick-Handler
        self.link_label.bind("<Button-1>", self.on_help_link_clicked)

    # ------------------------------------------------------------------
    # Datei laden
    # ------------------------------------------------------------------
    def browse_source(self):
        path = filedialog.askopenfilename(
            title=self.t("filedialog_open_title"),
            filetypes=[("JSON", "*.json"), ("All Files", "*.*")]
        )
        if not path:
            return
        self.source_path.set(path)
        self.load_show(path)

    def load_show(self, path: str):
        self.log(self.t("log_loading_show", path=path))
        try:
            with open(path, "r", encoding="utf-8") as f:
                data = json.load(f)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_load_title"),
                self.t("msg_error_load_file", error=e)
            )
            self.show_data = None
            self.show_obj = None
            self.btn_save.config(state="disabled")
            return

        show = data.get("show")
        if not isinstance(show, dict):
            messagebox.showerror(
                self.t("msg_error_load_title"),
                self.t("msg_error_structure")
            )
            self.show_data = None
            self.show_obj = None
            self.btn_save.config(state="disabled")
            return

        pattern_cue = show.get("patternCue")
        if not isinstance(pattern_cue, list):
            messagebox.showwarning(
                self.t("msg_warn_no_cues_title"),
                self.t("msg_warn_no_cues")
            )

        # Metadaten übernehmen
        name = show.get("name", "")
        show_id_val = show.get("ID") or show.get("id") or ""
        desc = show.get("description", "")

        self.show_name.set(name if isinstance(name, str) else "")
        self.show_id.set(show_id_val if isinstance(show_id_val, str) else "")
        self.show_desc.set(desc if isinstance(desc, str) else "")

        self.show_data = data
        self.show_obj = show

        self.log(
            self.t(
                "log_show_loaded",
                name=self.show_name.get() or "(leer)",
                id=self.show_id.get() or "(leer)",
                count=len(pattern_cue) if isinstance(pattern_cue, list) else 0
            )
        )

        self.btn_save.config(state="normal")

    # ------------------------------------------------------------------
    # Speichern mit Mapping
    # ------------------------------------------------------------------
    def save_as(self):
        if not self.show_data or not self.show_obj:
            messagebox.showwarning(
                self.t("msg_warn_no_show_title"),
                self.t("msg_warn_no_show")
            )
            return

        # Mapping auslesen
        mapping = {}
        used_old = set()
        for new_idx in range(1, 5):
            try:
                old_idx = int(self.group_vars[new_idx].get())
            except ValueError:
                messagebox.showerror(
                    self.t("msg_error_mapping_title"),
                    self.t("msg_error_mapping_invalid")
                )
                return
            mapping[new_idx] = old_idx
            used_old.add(old_idx)

        # Prüfen, ob es eine echte Permutation ist
        if used_old != {1, 2, 3, 4}:
            if not messagebox.askyesno(
                self.t("msg_confirm_mapping_title"),
                self.t("msg_confirm_mapping_body")
            ):
                return

        self.log(self.t("log_mapping_header"))
        for new_idx in range(1, 5):
            self.log(self.t("log_mapping_line", new=new_idx, old=mapping[new_idx]))

        # Zieldatei wählen
        out_path = filedialog.asksaveasfilename(
            title=self.t("filedialog_save_title"),
            defaultextension=".json",
            filetypes=[("JSON", "*.json"), ("All Files", "*.*")]
        )
        if not out_path:
            return

        # Arbeitskopie des JSON erzeugen
        out_data = copy.deepcopy(self.show_data)
        out_show = out_data.get("show", {})

        # Metadaten aktualisieren
        new_name = self.show_name.get().strip()
        new_id = self.show_id.get().strip()
        new_desc = self.show_desc.get().strip()

        if new_name:
            out_show["name"] = new_name

        if new_id:
            # ID-Key bestimmen
            if "ID" in out_show:
                id_key = "ID"
            elif "id" in out_show:
                id_key = "id"
            else:
                id_key = "ID"
            out_show[id_key] = new_id

        if new_desc:
            out_show["description"] = new_desc

        # Mapping auf alle Cues anwenden
        pattern_cue = out_show.get("patternCue")
        if isinstance(pattern_cue, list):
            changed = self._apply_group_mapping_to_cues(pattern_cue, mapping)
        else:
            changed = 0

        # Datei schreiben
        try:
            with open(out_path, "w", encoding="utf-8") as f:
                json.dump(out_data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_save_title"),
                self.t("msg_error_save_body", error=e)
            )
            return

        self.log(
            self.t(
                "log_save_done",
                path=out_path,
                changed=changed
            )
        )
        messagebox.showinfo(
            self.t("msg_info_done_title"),
            self.t("msg_info_done_body", changed=changed)
        )

    def _apply_group_mapping_to_cues(self, cue_list, mapping):
        """
        Wendet das Mapping auf alle Cues an.
        mapping[new_idx] = old_idx   (jeweils 1..4)
        """
        changed = 0
        for cue in cue_list:
            # Original-Gruppen auslesen
            old_groups = {}
            for idx in range(1, 5):
                key = GROUP_KEYS[idx]
                if key in cue:
                    old_groups[idx] = cue[key]
                else:
                    old_groups[idx] = None

            # neue Gruppen nach Mapping zusammensetzen
            new_groups = {}
            for new_idx in range(1, 5):
                old_idx = mapping[new_idx]
                new_groups[new_idx] = old_groups.get(old_idx)

            # Nur wenn sich etwas verändert hat, schreiben wir zurück
            if any(old_groups[i] is not new_groups[i] for i in range(1, 5)):
                changed += 1

            # Zurück in die Cue-Struktur schreiben
            for idx in range(1, 5):
                key = GROUP_KEYS[idx]
                val = new_groups[idx]
                if val is not None:
                    cue[key] = val
                elif key in cue:
                    # Wenn kein Wert mehr, Key ggf. löschen
                    del cue[key]

        return changed

    # ------------------------------------------------------------------
    # Logging-Helfer
    # ------------------------------------------------------------------
    def log(self, msg: str):
        self.log_text.insert(tk.END, msg + "\n")
        self.log_text.see(tk.END)


def main():
    app = MaestroGroupRemapGUI()
    app.mainloop()


if __name__ == "__main__":
    main()
