#!/usr/bin/env python3
import json
import os
import copy
import tkinter as tk
from tkinter import filedialog, messagebox, scrolledtext
import webbrowser


def load_language_file(path):
    """Lädt eine einfache key=value Sprachdatei."""
    data = {}
    if not os.path.exists(path):
        return data
    with open(path, "r", encoding="utf-8") as f:
        for line in f:
            line = line.strip()
            if not line or line.startswith("#"):
                continue
            if "=" not in line:
                continue
            key, value = line.split("=", 1)
            data[key.strip()] = value.strip()
    return data


class MaestroDurationTransitionTool(tk.Tk):
    def __init__(self):
        super().__init__()

        # Basisverzeichnis
        base_dir = os.path.dirname(os.path.abspath(__file__))

        # Pfad zur Config-Datei (Sprache)
        self.config_path = os.path.join(base_dir, "config.cfg")

        # Sprachdateien
        self.lang_files = {
            "de": os.path.join(base_dir, "de.cfg"),
            "en": os.path.join(base_dir, "en.cfg"),
            "fr": os.path.join(base_dir, "fr.cfg"),
            "es": os.path.join(base_dir, "es.cfg"),
        }
        self.languages = {
            code: load_language_file(path)
            for code, path in self.lang_files.items()
        }

        # zuletzt verwendete Sprache laden
        last_lang = self._load_last_language()
        if last_lang in self.languages:
            self.current_lang = last_lang
        elif "de" in self.languages:
            self.current_lang = "de"
        else:
            self.current_lang = next(iter(self.languages)) if self.languages else "de"

        self.tr = self.languages.get(self.current_lang, {})



        self.source_path = tk.StringVar()
        self.show_data = None
        self.show_obj = None  # self.show_data["show"]
        self.cue_list = None  # show["patternCue"]

        # Show-Metadaten
        self.show_name = tk.StringVar()
        self.show_id = tk.StringVar()
        self.show_desc = tk.StringVar()

        # Steuerung für Dauer / Transition
        self.duration_mode = tk.StringVar(value="keep")          # keep | infinite | fixed
        self.duration_fixed = tk.StringVar(value="60")

        self.trans_dur_mode = tk.StringVar(value="keep")         # keep | zero | fixed
        self.trans_dur_fixed = tk.StringVar(value="2")

        self.trans_type_mode = tk.StringVar(value="keep")        # keep | set | delete
        self.trans_type_value = tk.StringVar(value="NONE")

        self._build_widgets()
        self.apply_translations()

    # -------------------------------------------------------------------------
    # Config Sprache
    # -------------------------------------------------------------------------
    def _load_last_language(self):
        """Liest die zuletzt gewählte Sprache aus config.cfg (language=xx)."""
        try:
            if not os.path.exists(self.config_path):
                return None
            with open(self.config_path, "r", encoding="utf-8") as f:
                for line in f:
                    line = line.strip()
                    if not line or line.startswith("#"):
                        continue
                    if "=" not in line:
                        continue
                    key, value = line.split("=", 1)
                    if key.strip().lower() == "language":
                        return value.strip()
        except Exception:
            return None
        return None

    def _save_last_language(self):
        """Speichert die aktuell gewählte Sprache in config.cfg."""
        try:
            with open(self.config_path, "w", encoding="utf-8") as f:
                f.write(f"language={self.current_lang}\n")
        except Exception:
            # Tool soll auch laufen, wenn Config nicht geschrieben werden kann.
            pass

    # -------------------------------------------------------------------------
    # Übersetzungshilfen
    # -------------------------------------------------------------------------
    def t(self, msg_key, **kwargs):
        """Übersetzung holen, bei Bedarf mit .format(**kwargs)."""
        text = self.tr.get(msg_key, msg_key)
        # \n aus Sprachdatei in echte Zeilenumbrüche umwandeln
        text = text.replace("\\n", "\n")
        if kwargs:
            try:
                text = text.format(**kwargs)
            except Exception:
                pass
        return text


    def set_language(self, lang_code):
        if lang_code not in self.languages:
            return
        self.current_lang = lang_code
        self.tr = self.languages.get(lang_code, {})
        if hasattr(self, "lang_var"):
            self.lang_var.set(lang_code)
        self.apply_translations()
        self._save_last_language()

    def apply_translations(self):
        # Fenster-Titel
        self.title(self.t("app_title_duration"))

        # Pfad / Hinweis
        self.lbl_source_file.config(text=self.t("label_source_show_json_duration"))
        self.hint_label.config(text=self.t("hint_duration_info"))

        # Sprachlabel
        if self.current_lang == "de":
            self.lbl_lang.config(text="Sprache:")
        else:
            self.lbl_lang.config(text="Language:")

        # Show-Metadaten
        self.frame_meta.config(text=self.t("frame_meta_title_duration"))
        self.lbl_show_name.config(text=self.t("show_name_label"))
        self.lbl_show_id.config(text=self.t("show_id_label"))
        self.lbl_show_desc.config(text=self.t("show_desc_label"))

        # Globale Einstellungen
        self.frame_settings.config(text=self.t("frame_settings_title"))

        self.frame_dur.config(text=self.t("frame_dur_title"))
        self.rb_dur_keep.config(text=self.t("dur_keep"))
        self.rb_dur_inf.config(text=self.t("dur_infinite"))
        self.rb_dur_fixed.config(text=self.t("dur_fixed_label"))
        self.lbl_dur_hint.config(text=self.t("dur_hint"))

        self.frame_trans_dur.config(text=self.t("frame_trans_dur_title"))
        self.rb_td_keep.config(text=self.t("trans_dur_keep"))
        self.rb_td_zero.config(text=self.t("trans_dur_zero"))
        self.rb_td_fixed.config(text=self.t("trans_dur_fixed_label"))

        self.frame_trans_type.config(text=self.t("frame_trans_type_title"))
        self.rb_tt_keep.config(text=self.t("trans_type_keep"))
        self.rb_tt_set.config(text=self.t("trans_type_set_label"))
        self.rb_tt_del.config(text=self.t("trans_type_delete"))
        self.lbl_tt_hint.config(text=self.t("trans_type_hint"))

        # Cue-Übersicht & Log
        self.frame_overview.config(text=self.t("frame_overview_title"))
        self.frame_log.config(text=self.t("frame_log_title"))

        # Buttons
        self.btn_apply_save.config(text=self.t("button_apply_save"))
        self.btn_quit.config(text=self.t("button_quit"))

        # Footer
        self.author_label.config(text=self.t("footer_author_duration"))
        self.link_label.config(text=self.t("help_link_label"))

    # -------------------------------------------------------------------------
    # Help-Link
    # -------------------------------------------------------------------------
    def on_help_link_clicked(self, event=None):
        url = self.t("help_link_url")
        if url and url.startswith("http"):
            webbrowser.open(url)

    # ------------------------------------------------------------------
    # GUI-Aufbau
    # ------------------------------------------------------------------
    def _build_widgets(self):
        # Obere Zeile: Datei + Sprache
        frame_paths = tk.Frame(self)
        frame_paths.pack(fill="x", padx=10, pady=10)

        self.lbl_source_file = tk.Label(frame_paths, text="")
        self.lbl_source_file.grid(row=0, column=0, sticky="w")

        tk.Entry(frame_paths, textvariable=self.source_path, width=60).grid(
            row=0, column=1, padx=5
        )
        tk.Button(
            frame_paths,
            text="…",
            width=3,
            command=self.browse_source
        ).grid(row=0, column=2, padx=5)

        # Sprachwahl (rechts oben)
        self.lang_var = tk.StringVar(value=self.current_lang)
        self.lbl_lang = tk.Label(frame_paths, text="Sprache:")
        self.lbl_lang.grid(row=0, column=3, sticky="e", padx=(20, 5))

        lang_options = list(self.languages.keys())
        self.opt_lang = tk.OptionMenu(frame_paths, self.lang_var, *lang_options, command=self.set_language)
        self.opt_lang.grid(row=0, column=4, sticky="w")

        # Hinweis
        self.hint_label = tk.Label(
            self,
            text="",
            fg="gray",
            justify="left",
            anchor="w",
        )
        self.hint_label.pack(fill="x", padx=10, pady=(0, 10))

        # Show-Metadaten
        self.frame_meta = tk.LabelFrame(self, text="", padx=5, pady=5)
        self.frame_meta.pack(fill="x", padx=10, pady=5)

        self.lbl_show_name = tk.Label(self.frame_meta, text="")
        self.lbl_show_name.grid(row=0, column=0, sticky="w", padx=5, pady=2)
        tk.Entry(self.frame_meta, textvariable=self.show_name, width=50).grid(
            row=0, column=1, sticky="we", padx=5, pady=2
        )

        self.lbl_show_id = tk.Label(self.frame_meta, text="")
        self.lbl_show_id.grid(row=1, column=0, sticky="w", padx=5, pady=2)
        tk.Entry(self.frame_meta, textvariable=self.show_id, width=50).grid(
            row=1, column=1, sticky="we", padx=5, pady=2
        )

        self.lbl_show_desc = tk.Label(self.frame_meta, text="")
        self.lbl_show_desc.grid(row=2, column=0, sticky="w", padx=5, pady=2)
        tk.Entry(self.frame_meta, textvariable=self.show_desc, width=50).grid(
            row=2, column=1, sticky="we", padx=5, pady=2
        )

        self.frame_meta.columnconfigure(1, weight=1)

        # Einstellungen für Dauer & Transition
        self.frame_settings = tk.LabelFrame(self, text="", padx=5, pady=5)
        self.frame_settings.pack(fill="x", padx=10, pady=5)

        # Dauer (duration)
        self.frame_dur = tk.LabelFrame(self.frame_settings, text="", padx=5, pady=5)
        self.frame_dur.grid(row=0, column=0, sticky="nw", padx=5, pady=5)

        self.rb_dur_keep = tk.Radiobutton(
            self.frame_dur,
            text="",
            variable=self.duration_mode,
            value="keep"
        )
        self.rb_dur_keep.pack(anchor="w")

        self.rb_dur_inf = tk.Radiobutton(
            self.frame_dur,
            text="",
            variable=self.duration_mode,
            value="infinite"
        )
        self.rb_dur_inf.pack(anchor="w")

        frame_dur_fixed = tk.Frame(self.frame_dur)
        frame_dur_fixed.pack(anchor="w", pady=(5, 0))
        self.rb_dur_fixed = tk.Radiobutton(
            frame_dur_fixed,
            text="",
            variable=self.duration_mode,
            value="fixed"
        )
        self.rb_dur_fixed.pack(side="left", anchor="w")
        tk.Entry(frame_dur_fixed, textvariable=self.duration_fixed, width=6).pack(
            side="left", padx=5
        )

        self.lbl_dur_hint = tk.Label(
            self.frame_dur,
            text="",
            fg="gray"
        )
        self.lbl_dur_hint.pack(anchor="w", pady=(5, 0))

        # Transition Duration
        self.frame_trans_dur = tk.LabelFrame(self.frame_settings, text="", padx=5, pady=5)
        self.frame_trans_dur.grid(row=0, column=1, sticky="nw", padx=5, pady=5)

        self.rb_td_keep = tk.Radiobutton(
            self.frame_trans_dur,
            text="",
            variable=self.trans_dur_mode,
            value="keep"
        )
        self.rb_td_keep.pack(anchor="w")

        self.rb_td_zero = tk.Radiobutton(
            self.frame_trans_dur,
            text="",
            variable=self.trans_dur_mode,
            value="zero"
        )
        self.rb_td_zero.pack(anchor="w")

        frame_trans_dur_fixed = tk.Frame(self.frame_trans_dur)
        frame_trans_dur_fixed.pack(anchor="w", pady=(5, 0))
        self.rb_td_fixed = tk.Radiobutton(
            frame_trans_dur_fixed,
            text="",
            variable=self.trans_dur_mode,
            value="fixed"
        )
        self.rb_td_fixed.pack(side="left", anchor="w")
        tk.Entry(frame_trans_dur_fixed, textvariable=self.trans_dur_fixed, width=6).pack(
            side="left", padx=5
        )

        # Transition Typ
        self.frame_trans_type = tk.LabelFrame(self.frame_settings, text="", padx=5, pady=5)
        self.frame_trans_type.grid(row=0, column=2, sticky="nw", padx=5, pady=5)

        self.rb_tt_keep = tk.Radiobutton(
            self.frame_trans_type,
            text="",
            variable=self.trans_type_mode,
            value="keep"
        )
        self.rb_tt_keep.pack(anchor="w")

        frame_trans_type_set = tk.Frame(self.frame_trans_type)
        frame_trans_type_set.pack(anchor="w", pady=(5, 0))
        self.rb_tt_set = tk.Radiobutton(
            frame_trans_type_set,
            text="",
            variable=self.trans_type_mode,
            value="set"
        )
        self.rb_tt_set.pack(side="left", anchor="w")
        tk.Entry(frame_trans_type_set, textvariable=self.trans_type_value, width=15).pack(
            side="left", padx=5
        )

        self.rb_tt_del = tk.Radiobutton(
            self.frame_trans_type,
            text="",
            variable=self.trans_type_mode,
            value="delete"
        )
        self.rb_tt_del.pack(anchor="w", pady=(5, 0))

        self.lbl_tt_hint = tk.Label(
            self.frame_trans_type,
            text="",
            fg="gray"
        )
        self.lbl_tt_hint.pack(anchor="w", pady=(5, 0))

        # Mittlerer Bereich: Cue-Übersicht
        frame_middle = tk.Frame(self)
        frame_middle.pack(fill="both", expand=True, padx=10, pady=5)

        self.frame_overview = tk.LabelFrame(frame_middle, text="", padx=5, pady=5)
        self.frame_overview.pack(side="left", fill="both", expand=True)

        self.cue_text = scrolledtext.ScrolledText(self.frame_overview, wrap="none", height=14)
        self.cue_text.pack(fill="both", expand=True)

        # Rechter Bereich: Log & Buttons
        frame_right = tk.Frame(frame_middle)
        frame_right.pack(side="right", fill="y", padx=(5, 0))

        self.frame_log = tk.LabelFrame(frame_right, text="", padx=5, pady=5)
        self.frame_log.pack(fill="both", expand=True)

        self.log_text = scrolledtext.ScrolledText(self.frame_log, wrap="word", height=8, width=35)
        self.log_text.pack(fill="both", expand=True)

        frame_buttons = tk.Frame(frame_right)
        frame_buttons.pack(fill="x", pady=(5, 0))

        self.btn_apply_save = tk.Button(
            frame_buttons,
            text="",
            command=self.apply_and_save,
            state="disabled"
        )
        self.btn_apply_save.pack(fill="x", pady=(0, 5))

        self.btn_quit = tk.Button(
            frame_buttons,
            text="",
            command=self.destroy
        )
        self.btn_quit.pack(fill="x")

        # Footer
        footer = tk.Frame(self)
        footer.pack(fill="x", padx=10, pady=(0, 10))

        self.author_label = tk.Label(
            footer,
            text="",
            fg="gray"
        )
        self.author_label.pack(side="left", anchor="w")

        self.link_label = tk.Label(
            footer,
            text="",
            fg="blue",
            cursor="hand2"
        )
        self.link_label.pack(side="right", anchor="e")

        self.link_label.bind("<Button-1>", self.on_help_link_clicked)

    # ------------------------------------------------------------------
    # Datei laden
    # ------------------------------------------------------------------
    def browse_source(self):
        path = filedialog.askopenfilename(
            title=self.t("filedialog_open_title"),
            filetypes=[("JSON", "*.json"), ("All Files", "*.*")]
        )
        if not path:
            return
        self.source_path.set(path)
        self.load_show(path)

    def load_show(self, path: str):
        self.log(self.t("log_loading_show", path=path))
        self.cue_text.delete("1.0", tk.END)

        try:
            with open(path, "r", encoding="utf-8") as f:
                data = json.load(f)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_load_title"),
                self.t("msg_error_load_file", error=e)
            )
            self.show_data = None
            self.show_obj = None
            self.cue_list = None
            self.btn_apply_save.config(state="disabled")
            return

        show = data.get("show")
        if not isinstance(show, dict):
            messagebox.showerror(
                self.t("msg_error_load_title"),
                self.t("msg_error_structure")
            )
            self.show_data = None
            self.show_obj = None
            self.cue_list = None
            self.btn_apply_save.config(state="disabled")
            return

        cues = show.get("patternCue")
        if not isinstance(cues, list):
            messagebox.showwarning(
                self.t("msg_warn_no_cues_title"),
                self.t("msg_warn_no_cues")
            )
            cues = []

        # Metadaten übernehmen
        name = show.get("name", "")
        show_id_val = show.get("ID") or show.get("id") or ""
        desc = show.get("description", "")

        self.show_name.set(name if isinstance(name, str) else "")
        self.show_id.set(show_id_val if isinstance(show_id_val, str) else "")
        self.show_desc.set(desc if isinstance(desc, str) else "")

        self.show_data = data
        self.show_obj = show
        self.cue_list = cues

        self.log(
            self.t(
                "log_show_loaded",
                name=self.show_name.get() or "(leer)",
                id=self.show_id.get() or "(leer)",
                count=len(self.cue_list)
            )
        )

        self._fill_cue_overview()

        if self.cue_list:
            self.btn_apply_save.config(state="normal")
        else:
            self.btn_apply_save.config(state="disabled")

    def _fill_cue_overview(self):
        self.cue_text.delete("1.0", tk.END)
        if not self.cue_list:
            self.cue_text.insert(tk.END, "Keine Cues vorhanden.\n")
            return

        for idx, cue in enumerate(self.cue_list):
            name = cue.get("name", "<ohne Name>")
            duration = cue.get("duration", "")
            trans = cue.get("transition", {})
            if not isinstance(trans, dict):
                trans = {}
            t_type = trans.get("type", "<kein type>")
            t_dur = trans.get("duration", "<kein duration>")

            line = (
                f"[{idx}] {name} | "
                f"duration={duration}, transition.type={t_type}, transition.duration={t_dur}\n"
            )
            self.cue_text.insert(tk.END, line)


    # ------------------------------------------------------------------
    # Anwenden & Speichern
    # ------------------------------------------------------------------
    def apply_and_save(self):
        if not (self.show_data and self.show_obj and self.cue_list is not None):
            messagebox.showwarning(
                self.t("msg_warn_no_show_title"),
                self.t("msg_warn_no_show")
            )
            return

        # Zielpfad
        out_path = filedialog.asksaveasfilename(
            title=self.t("filedialog_save_title"),
            defaultextension=".json",
            filetypes=[("JSON", "*.json"), ("All Files", "*.*")]
        )
        if not out_path:
            return

        # Einstellungen interpretieren
        try:
            dur_mode = self.duration_mode.get()
            trans_dur_mode = self.trans_dur_mode.get()
            trans_type_mode = self.trans_type_mode.get()

            if dur_mode == "fixed":
                dur_fixed = int(self.duration_fixed.get())
            elif dur_mode == "infinite":
                dur_fixed = -1
            else:
                dur_fixed = None  # keep

            if trans_dur_mode == "fixed":
                trans_dur_fixed = int(self.trans_dur_fixed.get())
            elif trans_dur_mode == "zero":
                trans_dur_fixed = 0
            else:
                trans_dur_fixed = None  # keep

            if trans_type_mode == "set":
                t_type_val = self.trans_type_value.get().strip()
                if not t_type_val:
                    messagebox.showerror(
                        self.t("msg_error_type_title"),
                        self.t("msg_error_type_empty")
                    )
                    return
            else:
                t_type_val = None

        except ValueError:
            messagebox.showerror(
                self.t("msg_error_number_title"),
                self.t("msg_error_number_body")
            )
            return

        # Arbeitskopie der Show
        out_data = copy.deepcopy(self.show_data)
        out_show = out_data.get("show", {})
        out_cues = out_show.get("patternCue", [])

        # Metadaten aktualisieren
        new_name = self.show_name.get().strip()
        new_id = self.show_id.get().strip()
        new_desc = self.show_desc.get().strip()

        if new_name:
            out_show["name"] = new_name

        if new_id:
            if "ID" in out_show:
                id_key = "ID"
            elif "id" in out_show:
                id_key = "id"
            else:
                id_key = "ID"
            out_show[id_key] = new_id

        if new_desc:
            out_show["description"] = new_desc

        # Änderungen auf alle Cues anwenden
        changed = 0
        for cue in out_cues:
            orig_cue = repr(cue)  # grober Vergleich

            # duration
            if dur_mode in ("fixed", "infinite"):
                cue["duration"] = dur_fixed

            # transition
            trans = cue.get("transition")
            if not isinstance(trans, dict):
                trans = {}
            # duration in transition
            if trans_dur_mode in ("fixed", "zero"):
                trans["duration"] = trans_dur_fixed
            # type
            if trans_type_mode == "set":
                trans["type"] = t_type_val
            elif trans_type_mode == "delete":
                if "type" in trans:
                    del trans["type"]

            # Nur wenn transition überhaupt Felder hat, sinnvoll setzen
            if trans:
                cue["transition"] = trans

            if repr(cue) != orig_cue:
                changed += 1

        # Datei schreiben
        try:
            with open(out_path, "w", encoding="utf-8") as f:
                json.dump(out_data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            messagebox.showerror(
                self.t("msg_error_save_title"),
                self.t("msg_error_save_body", error=e)
            )
            return

        self.log(
            self.t(
                "log_save_done_duration",
                path=out_path,
                changed=changed
            )
        )
        messagebox.showinfo(
            self.t("msg_info_done_title"),
            self.t("msg_info_done_body_duration", changed=changed)
        )

    # ------------------------------------------------------------------
    # Logging-Helfer
    # ------------------------------------------------------------------
    def log(self, msg: str):
        self.log_text.insert(tk.END, msg + "\n")
        self.log_text.see(tk.END)


def main():
    app = MaestroDurationTransitionTool()
    app.mainloop()


if __name__ == "__main__":
    main()


